/*************************************************************************************************************
 *                                                                                                           *
 *  Copyright (c) 2015, Intel Corporation                                                                    *
 *                                                                                                           *
 *  Redistribution and use in source and binary forms, with or without                                       *
 *  modification, are permitted provided that the following conditions are met:                              *
 *                                                                                                           *
 *      * Redistributions of source code must retain the above copyright notice,                             *
 *        this list of conditions and the following disclaimer.                                              *
 *      * Redistributions in binary form must reproduce the above copyright                                  *
 *        notice, this list of conditions and the following disclaimer in the                                *
 *        documentation and/or other materials provided with the distribution.                               *
 *      * Neither the name of Intel Corporation nor the names of its contributors                            *
 *        may be used to endorse or promote products derived from this software                              *
 *        without specific prior written permission.                                                         *
 *                                                                                                           *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"                              *
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE                                *
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                           *
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE                              *
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL                               *
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR                               *
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER                               *
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,                            *
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE                            *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                     *
 *                                                                                                           *
 *************************************************************************************************************
 *                                                                                                           *
 *  Module name:                                                                                             *
 *      freebsdnaldriver.c                                                                                   *
 *                                                                                                           *
 *  Abstract:                                                                                                *
 *      This file contains sources for functions that handles the NAL driver                                 *
 *                                                                                                           *
 ************************************************************************************************************/

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <nalioctl.h>
#include <freebsdnalioctl.h>
#include <freebsdnaldriver.h>
#include <freebsddriverdevice_i.h>
#include <freebsdos_i.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>

/***************************************************************************
 * Local variables definitions
 ***************************************************************************/
struct mtx                  Global_AtomicTestSetSpinLock;
NAL_ADAPTER_IN_USE_TABLE    Global_AdapterInUse[NAL_DRIVER_MAX_ADAPTERS];
UINT32                      Global_DriverReferenceCount = 0;
struct cdev *               Global_NalDevicePointer     = NULL;
struct cdevsw               Global_DeviceEntryPoints    = {
        .d_version  = D_VERSION ,           /* version of kernel driver mode */
        .d_name     = "nal" ,               /* name of the driver in /dev/ directory */
        .d_open     = NalOpenDriver ,       /* handler for opening driver */
        .d_close    = NalReleaseDriver ,    /* handler for closing driver */
        .d_ioctl    = NalHandleIoctl        /* handler for ioctl calls */
};

/* Definition of module driver */
DEV_MODULE( nal , NalHandleModuleEvent , NULL );

/***************************************************************************
**
** Name:            NalHandleModuleEvent()
**
** Description:     This is the function, that handles driver module events.
**                  It loads and unloads the nal driver module.
**
** Arguments:       Module  - Pointer to module structure
**                  Event   - Event id to handle
**                  Argument-
**
** Returns:         Code of error from errno.h
**                      0           - when success
**                      EDOOFUS     - when there is error in function implementation
**                      EOPNOTSUPP  - when an event is not recognized
**                      ENXIO       - when device was not loaded before
**
****************************************************************************/
int
NalHandleModuleEvent (
                      IN      module_t    Module ,
                      IN      int         Event ,
                      IN OUT  void *      Argument )
{
  int ErrorCode = EDOOFUS;  /* EDOOFUS - programming error */

  switch(Event)
  {
  /***************************************************************************
   *
   * This case is executed when an event is not recognized
   *
   ***************************************************************************/
  default:
       ErrorCode = EOPNOTSUPP;  /* EOPNOTSUPP - operation not supported */
       break;
  /***************************************************************************
    *
    * This case is executed when the system is about to shut down.
    *
   ***************************************************************************/
  case MOD_SHUTDOWN:
      ErrorCode = EOPNOTSUPP;  /* EOPNOTSUPP - operation not supported */
      break;
  /***************************************************************************
   *
   * This case is executed when the NAL driver should be loaded to the system.
   * It initializes module variables to work.
   *
   ***************************************************************************/
  case MOD_LOAD:
       uprintf(
               "Intel Pro Diagnostic Driver loading %d.%d.%d.%d\n" ,
               FREEBSD_DRIVER_MAJOR_VERSION  ,
               FREEBSD_DRIVER_MINOR_VERSION ,
               FREEBSD_DRIVER_BUILD_VERSION ,
               FREEBSD_DRIVER_FIX_VERSION );

       /* Clearing global arrays */
       memset(Global_PciAllocationSlotsTable, 0 , sizeof(Global_PciAllocationSlotsTable));
       memset(Global_DmaAllocationSlotsTable, 0 , sizeof(Global_DmaAllocationSlotsTable));
       memset(&Global_AtomicTestSetSpinLock , 0 , sizeof(Global_AtomicTestSetSpinLock));
       memset(Global_AdapterInUse           , 0 , sizeof(Global_AdapterInUse));

       mtx_init(&Global_AtomicTestSetSpinLock , "AtomicTestSetSpinLock" , NULL , MTX_DEF);

       /* The function below MUST be always AT THE END of module loading. */
       Global_NalDevicePointer = make_dev(
                                          &Global_DeviceEntryPoints ,   /* Driver interfaces */
                                          0 ,                           /* Index of module */
                                          UID_ROOT ,                    /* User-ID (create device as root user) */
                                          GID_WHEEL,                    /* Group-ID */
                                          0600 ,                        /* Permissions for device (only for root) */
                                          "nal" );                      /* Name of device */

       ErrorCode = 0;
       break;
   /***************************************************************************
   *
   * This case is executed when the NAL driver should be unloaded from the system.
   * It destroys all variables, free all allocated memory
   *
   ***************************************************************************/
  case MOD_UNLOAD:
       do
       {
           if(Global_NalDevicePointer == NULL)
           {
               /* Device not configured */
               ErrorCode = ENXIO;
               break;
           }

           if(Global_DriverReferenceCount > 0)
           {
               uprintf("Intel Pro Diagnostic Driver version %d.%d.%d.%d exiting with %d %s\n" ,
                       FREEBSD_DRIVER_MAJOR_VERSION  ,
                       FREEBSD_DRIVER_MINOR_VERSION ,
                       FREEBSD_DRIVER_BUILD_VERSION ,
                       FREEBSD_DRIVER_FIX_VERSION ,
                       Global_DriverReferenceCount ,
                       (Global_DriverReferenceCount==1) ? "reference" : "references");
           }

           /* The function below MUST be called before any unloading operations */
           destroy_dev(Global_NalDevicePointer);

           mtx_destroy(&Global_AtomicTestSetSpinLock);

           Global_NalDevicePointer  = NULL;
           ErrorCode                = 0;

       } while(0);
       break;
  }

  return ErrorCode;
}

/***************************************************************************
**
** Name:            NalOpenDriver()
**
** Description:     The function prepares the driver to perform ioctl
**                  commands.
**
** Arguments:       Device      - Pointer to device structure
**                  OpenFlags   - Flags of open mode
**                  DeviceType  - Type of the device
**                  Thread      - Pointer to a thread
**
** Returns:         Code of error from errno.h
**                      0           - when success
**
****************************************************************************/
int
NalOpenDriver(
              IN    struct cdev *   Device ,
              IN    int             OpenFlags ,
              IN    int             DeviceType ,
              IN    struct thread * Thread )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalReleaseDriver()
**
** Description:     The function releases driver. It is called, when it
**                  should be closed.
**
** Arguments:       Device      - Pointer to device structure
**                  ReleaseFlags- Flags of close mode
**                  DeviceType  - Type of the device
**                  Thread      - Pointer to a thread
**
** Returns:         Code of error from errno.h
**                      0           - when success
**
****************************************************************************/
int
NalReleaseDriver(
                 IN     struct cdev *   Device ,
                 IN     int             ReleaseFlags  ,
                 IN     int             DeviceType ,
                 IN     struct thread * Thread )
{
    _NalDriverDecrementReferenceCount();
    return 0;
}

/***************************************************************************
**
** Name:            NalHandleIoctl()
**
** Description:     The function for handling IOCTL commands for NAL device.
**
** Arguments:       Device      - Pointer to device structure
**                  Command     - IOCTL command to execute
**                  Data        - Pointer to the data
**                  Flags       - Flags the command
**                  Thread      - Pointer to a thread
**
** Returns:         Code of error from errno.h
**                      0           - when success
**                      EDOOFUS     - when there is error in function implementation
**                      EOPNOTSUPP  - when the command is not recognized
**
****************************************************************************/
int
NalHandleIoctl(
               IN      struct cdev *   Device ,
               IN      u_long          Command ,
               IN OUT  caddr_t         Data ,
               IN      int             Flags ,
               IN      struct thread * Thread )
{
    int                   ErrorCode         = EDOOFUS;
    NAL_IOCTL_INPUT_DATA  InputData         = {};
    NAL_IOCTL_INPUT_DATA* NalIoctlInputData = (NAL_IOCTL_INPUT_DATA*)Data;

    /*************************************************************************
     *
     * The Data is not copied because the ioctl command has assigned size of
     * data, that is copied before call of this function
     *
     ************************************************************************/
    memset(&InputData , 0 , sizeof(InputData));

    switch(Command)
    {
    /*************************************************************************
     *
     * This case must be first. It occurs when the operation is not supported.
     *
     ************************************************************************/
    default:
        ErrorCode   = EOPNOTSUPP;
        break;
    /*************************************************************************
     *
     * Received OSI-type command
     *
     ************************************************************************/
    case IOCTL_NAL_OSI:
        NalResolveOsiIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;
    /*************************************************************************
     *
     * Received HW-BUS-type command
     *
     ************************************************************************/
    case IOCTL_NAL_HW_BUS:
        NalResolveHwBusIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;
    /*************************************************************************
     *
     * Received NDI-type command
     *
     ************************************************************************/
    case IOCTL_NAL_NDI:
        NalResolveNdiIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;

    /*************************************************************************
     *
     * Received FreeBSD specific command
     *
     ************************************************************************/
    case IOCTL_NAL_OS_SPECIFIC:
        NalResolveOsSpecificIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;
    }

    return ErrorCode;
}

/***************************************************************************
**
** Name:            _NalMarkAdapterInUse()
**
** Description:     This attempts to mark the adapter in use if the init flags are not
**                  shared. It returns if the driver was already marked in use and if
**                  the driver can be used.
**
** Arguments:       NalDevice   = NAL_DEVICE_LOCATION with the PCI or BUS device info.
**                  Lock        = TRUE to lock the device, FALSE to unlock it.
**
** Returns:         TRUE  = Device is in use
**                  FALSE = Device is not yet in use.
**
****************************************************************************/
BOOLEAN
_NalMarkAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice,
    IN  BOOLEAN               Lock
    )
{
    UINTN   i         = 0;
    BOOLEAN CanBeUsed = FALSE;

    /* First scan all adapters in memory to see if any match the NalDevice requested */
    for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
    {
        /* Check to see if this adapter slot matches our NalDevice */
        if(Global_AdapterInUse[i].DeviceLocation.Reserved == NalDevice.Reserved)
        {
            /* See if the matching device is already marked "In Use" */
            if(Global_AdapterInUse[i].InUse == TRUE)
            {
                /* The device is in use, determine if we're being asked to "unlock" it */
                if(Lock == FALSE)
                {
                    Global_AdapterInUse[i].InUse = FALSE;
                    Global_AdapterInUse[i].DeviceLocation.Reserved = 0;
                    CanBeUsed = TRUE;
                    break;
                }
                else
                {
                    CanBeUsed = FALSE;
                    break;
                }
            }

            /* Slot found, we're asking to mark in use, and the dev is not in use */
            else if(Lock == TRUE)
            {
                Global_AdapterInUse[i].InUse = TRUE;
                CanBeUsed = TRUE;
                break;
            }
        }
    }

    /* Entry not found and we're looking to add one. */
    if(i == NAL_DRIVER_MAX_ADAPTERS && Lock == TRUE)
    {
        /* Look for a free slot */
        for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
        {
            if(Global_AdapterInUse[i].DeviceLocation.Reserved == 0)
            {
                break;
            }
        }

        /* This only runs if a free slot was located */
        if(i<NAL_DRIVER_MAX_ADAPTERS)
        {
            Global_AdapterInUse[i].DeviceLocation.Reserved = NalDevice.Reserved;
            Global_AdapterInUse[i].InUse = Lock;
            CanBeUsed = TRUE;
        }
    }

    /* This happens if the adapter is requested to be unlocked but doesnt exist */
    else if(i == NAL_DRIVER_MAX_ADAPTERS && Lock == FALSE)
    {
        CanBeUsed = TRUE;
    }

    return CanBeUsed;
}

/***************************************************************************
**
** Name:            _NalIsAdapterInUse()
**
** Description:     This searches to see if the adapter is already locked as in use.
**
** Arguments:       NalDevice   = NAL_DEVICE_LOCATION with the PCI or BUS device info.
**
** Returns:         TRUE  = device can be used
**                  FALSE = device cannot be used.
**
****************************************************************************/
BOOLEAN
_NalIsAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice
    )
{
    UINTN   i         = 0;
    BOOLEAN IsInUse   = FALSE;

    /* First scan all adapters in memory to see if any match the NalDevice requested */
    for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
    {
        /* Check to see if this adapter slot matches our NalDevice */
        if(Global_AdapterInUse[i].DeviceLocation.Reserved == NalDevice.Reserved)
        {
            /* See if the matching device is already marked "In Use" */
            if(Global_AdapterInUse[i].InUse == TRUE)
            {
                IsInUse = TRUE;
                break;
            }
        }
    }

    return IsInUse;
}


/***************************************************************************
**
** Name:            _NalDriverGetReferenceCount()
**
** Description:     This returns the reference count of the number of user mode components
**                  have referenced the driver.
**
** Arguments:       None
**
** Returns:         Number of references
**
****************************************************************************/
UINT32
_NalDriverGetReferenceCount(
    VOID
    )
{
    return Global_DriverReferenceCount;
}

/***************************************************************************
**
** Name:            _NalDriverIncrementReferenceCount()
**
** Description:     This increments the reference count of the number of user mode components
**                  have referenced the driver.
**
** Arguments:       nothing
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalDriverIncrementReferenceCount(
    VOID
    )
{
    /* Note, do not user the atomic increment here because it's intended for ring3 and will perform
     * a copy_from_user and copy_to_user on the address */
    mtx_lock(&Global_AtomicTestSetSpinLock);
    Global_DriverReferenceCount++;
    mtx_unlock(&Global_AtomicTestSetSpinLock);
}

/***************************************************************************
**
** Name:            _NalDriverDecrementReferenceCount()
**
** Description:     This decrements the reference count of the number of user mode components
**                  have referenced the driver.
**
** Arguments:       nothing
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalDriverDecrementReferenceCount(
    VOID
    )
{
    /* Note, do not user the atomic increment here because it's intended for ring3 and will perform
     * a copy_from_user and copy_to_user on the address */
    mtx_lock(&Global_AtomicTestSetSpinLock);
    if(Global_DriverReferenceCount > 0)
    {
        Global_DriverReferenceCount--;
    }
    mtx_unlock(&Global_AtomicTestSetSpinLock);
}
