/*************************************************************************************************************
 *                                                                                                           *
 *  Copyright (c) 2015, Intel Corporation                                                                    *
 *                                                                                                           *
 *  Redistribution and use in source and binary forms, with or without                                       *
 *  modification, are permitted provided that the following conditions are met:                              *
 *                                                                                                           *
 *      * Redistributions of source code must retain the above copyright notice,                             *
 *        this list of conditions and the following disclaimer.                                              *
 *      * Redistributions in binary form must reproduce the above copyright                                  *
 *        notice, this list of conditions and the following disclaimer in the                                *
 *        documentation and/or other materials provided with the distribution.                               *
 *      * Neither the name of Intel Corporation nor the names of its contributors                            *
 *        may be used to endorse or promote products derived from this software                              *
 *        without specific prior written permission.                                                         *
 *                                                                                                           *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"                              *
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE                                *
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                           *
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE                              *
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL                               *
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR                               *
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER                               *
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,                            *
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE                            *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                     *
 *                                                                                                           *
 *************************************************************************************************************
 *                                                                                                           *
 *  Module name:                                                                                             *
 *      freebsdos_i.h                                                                                        *
 *                                                                                                           *
 *  Abstract:                                                                                                *
 *      This file contains definitions of interface for freebsd os.                                          *
 *                                                                                                           *
 ************************************************************************************************************/
#ifndef SDK_NAL_INC_FREEBSD_FREEBSDOS_I_H_
#define SDK_NAL_INC_FREEBSD_FREEBSDOS_I_H_

#ifndef NAL_DRIVER
#include <sys/time.h>
#include <stdio.h>
#include <stdarg.h>           /* for va_start etc             */
#include <string.h>
#include <ctype.h>
#endif

#include <naltypes.h>
#include <hwbus_t.h>

#define _NAL_LOG_FILE_PATH              "/var/log/iqvfreebsd.log"

#define NAL_FREEBSD_INTERRUPT_SIGNATURE   0xA5BABA5A

#define MS_DELAY_FOR_CALCULATION        100

#ifndef HIBYTE
    #define HIBYTE(_x) (UINT8)(((_x)>>8)&0xFF)
#endif

#ifndef LOBYTE
    #define LOBYTE(_x) (UINT8)((_x)&0xFF)
#endif

/* Data type manipulation macros. */
#ifndef HIWORD
    #define HIWORD(_x) ((UINT16)(((_x)>>16)&0xFFFF))
#endif

#ifndef LOWORD
    #define LOWORD(_x) ((UINT16)((_x)&0xFFFF))
#endif

#define HIDWORD(_x) ((UINT32)(((_x)>>32)&0xFFFFFFFF))
#define LODWORD(_x) ((UINT32)((_x)&0xFFFFFFFF))

#define LOW_BYTE(word)      LOBYTE(word)
#define HIGH_BYTE(word)     HIBYTE(word)

#define LOW_WORD(dword)     LOWORD(dword)
#define HIGH_WORD(dword)    HIWORD(dword)

#define MAKE_WORD(hi, low)                  \
    ((UINT16) ((((UINT16)(hi)) << 8) | (low)))

#define MAKE_DWORD(hi, low)                 \
    ((UINT32) ((((UINT32)(hi)) << 16) | (low)))

#define MAKE_QWORD(hi, low)                  \
    ((UINT64) ((((UINT64)(hi)) << 32) | (low)))

#if defined NAL_BIG_ENDIAN
/* byte swap a 2 byte value */
#define BYTE_SWAP_WORD(value)     (UINT16)( (((UINT16)(value) & 0x00ff)) | \
                                            (((UINT16)(value) & 0xff00)) )
/* Byte swap a 4 byte value */
#define BYTE_SWAP_DWORD(dword)    (UINT32)( (((UINT32)(dword) & 0x000000ff) << 24) |    \
                                            (((UINT32)(dword) & 0x0000ff00) << 8)  |    \
                                            (((UINT32)(dword) & 0x00ff0000) >> 8)  |    \
                                            (((UINT32)(dword) & 0xff000000) >> 24) )

/* Word swap a 4 byte value */
#define WORD_SWAP_DWORD(value)    (UINT32)( (((UINT32)(value) & 0x0000FFFF)) | \
                                            (((UINT32)(value) & 0xFFFF0000)))

/* Word swap a 8 byte value */
#define BYTE_SWAP_QWORD(_dest, _src)        \
{                                           \
    ((UINT8*)_dest)[0] = ((UINT8*)_src)[7]; \
    ((UINT8*)_dest)[1] = ((UINT8*)_src)[6]; \
    ((UINT8*)_dest)[2] = ((UINT8*)_src)[5]; \
    ((UINT8*)_dest)[3] = ((UINT8*)_src)[4]; \
    ((UINT8*)_dest)[4] = ((UINT8*)_src)[3]; \
    ((UINT8*)_dest)[5] = ((UINT8*)_src)[2]; \
    ((UINT8*)_dest)[6] = ((UINT8*)_src)[1]; \
    ((UINT8*)_dest)[7] = ((UINT8*)_src)[0]; \
}

/* Macro to byte swap a word. */
#define WORD_SWAP(dword) WORD_SWAP_DWORD(dword)

/* Macro to byte swap a word. */
#define BYTE_SWAP(word)  BYTE_SWAP_WORD(word)

#else
/* Macros to create:
 *   WORD from 2 bytes,
 *   DWORD from 2 words,
 *   QWORD from 2 dwords.
 */

/* byte swap a 2 byte value */
#define BYTE_SWAP_WORD(value)     (UINT16)( (((UINT16)(value) & 0x00ff) << 8) | \
                                            (((UINT16)(value) & 0xff00) >> 8) )

/* Byte swap a 4 byte value */
#define BYTE_SWAP_DWORD(dword)    (UINT32)( (((UINT32)(dword) & 0x000000ff) << 24) |    \
                                            (((UINT32)(dword) & 0x0000ff00) << 8)  |    \
                                            (((UINT32)(dword) & 0x00ff0000) >> 8)  |    \
                                            (((UINT32)(dword) & 0xff000000) >> 24) )

/* Word swap a 4 byte value */
#define WORD_SWAP_DWORD(value)    (UINT32)( (((UINT32)(value) & 0x0000FFFF) << 16) | \
                                            (((UINT32)(value) & 0xFFFF0000) >> 16) )

/* Word swap a 8 byte value */
#define BYTE_SWAP_QWORD(_dest, _src)        \
{                                           \
    ((UINT8*)_dest)[0] = ((UINT8*)_src)[7]; \
    ((UINT8*)_dest)[1] = ((UINT8*)_src)[6]; \
    ((UINT8*)_dest)[2] = ((UINT8*)_src)[5]; \
    ((UINT8*)_dest)[3] = ((UINT8*)_src)[4]; \
    ((UINT8*)_dest)[4] = ((UINT8*)_src)[3]; \
    ((UINT8*)_dest)[5] = ((UINT8*)_src)[2]; \
    ((UINT8*)_dest)[6] = ((UINT8*)_src)[1]; \
    ((UINT8*)_dest)[7] = ((UINT8*)_src)[0]; \
}

/* Macro to byte swap a word. */
#define WORD_SWAP(dword) WORD_SWAP_DWORD(dword)

/* Macro to byte swap a word. */
#define BYTE_SWAP(word)  BYTE_SWAP_WORD(word)
#endif

/* These 2 macros are used to make word for data sections when calculating checksum
 * They had to be defined separately to accommodate the case for big-endian machines
 * when checksum is calculated. The data section contains UIN8 bytes. These are treated
 * the same way whether the packet is host byte ordered or nw byte ordered.
 */
#define MAKE_DATA_WORD_TCP_CHECKSUM(hi, low) \
    ((UINT16) ((((UINT16)(hi)) << 8) | (low)))

#define MAKE_DATA_WORD_UDP_CHECKSUM(hi, low) \
    ((UINT16) ((((UINT16)(hi)) << 8) | (low)))

#define _NAL_MIN_MMAP_SIZE        65

#ifndef NAL_DRIVER
#include <freebsdnallibrary.h>
#endif

typedef struct _NAL_FREEBSD_ISR_DEVICE
{
    UINT32          Signature;
    BOOLEAN         DeviceInterrupted;
    KVOID*          HardwareVirtualAddress;
    UINT32          Irq;
    UINT32          MacType;
} NAL_FREEBSD_ISR_DEVICE;

#define NAL_FREEBSD_MAX_CONTIGUOUS_MEMORY_ALLOCATION    (128 * 1024)
#define NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS    50000

#ifndef NAL_DRIVER
//#if 1
//typedef pthread_t NAL_THREAD_ID;
#include <pthread.h>

typedef struct _NAL_FREEBSD_TIMER_OBJECT
{
    NAL_TIMER_CALLBACK  Callback;
    UINT32              ThreadId;
    struct itimerval    TimerVal;
    VOID*               Context;
} NAL_FREEBSD_TIMER_OBJECT;

typedef struct _NAL_FREEBSD_MEMORY_MAP_TABLE
{
    INT32                   ReferenceCount;
    VOID*                   VirtualAddress;
    VOID*                   AlignedVirtualAddress;
    NAL_PHYSICAL_ADDRESS    AlignedPhysicalAddress;
    UINT32                  Alignment;
    UINT32                  BytesAllocated;
} NAL_FREEBSD_MEMORY_MAP_TABLE;

typedef struct NAL_FREEBSD_NONPAGED_MEMORY_TABLE
{
    UINT32                  ReferenceCount;
    KVOID*                  VirtualAddress;
    VOID*                   MappedVirtualAddress;
} NAL_FREEBSD_NONPAGED_MEMORY_TABLE;

typedef struct _NAL_FREEBSD_DMA_PCI_MEMORY_TABLE
{
    UINT32                  ReferenceCount;
    KVOID*                  VirtualAddress;
    KVOID*                  KernelAddress;
    NAL_PHYSICAL_ADDRESS    PhysicalAddress;
    UINT32                  Size;
} NAL_FREEBSD_DMA_PCI_MEMORY_TABLE;

typedef struct _NAL_FREEBSD_THREAD_CONTEXT
{
        pthread_t           Thread;
        VOID*               Context;
        NAL_THREAD_FUNC     ThreadFunction;
        BOOLEAN             ThreadRunning;
} NAL_FREEBSD_THREAD_CONTEXT;

typedef NAL_FREEBSD_THREAD_CONTEXT NAL_THREAD_ID;

#else

#include <machine/bus.h>
#include <sys/param.h>
#include <sys/bus.h>

#include <machine/bus.h>
#include <sys/rman.h>
#include <machine/resource.h>

typedef struct _NAL_FREEBSD_PCI_ALLOCATION_SLOT
{
        KVOID*                      KernelAddress;
        NAL_PHYSICAL_ADDRESS        PhysicalAddress;
        struct resource *           Resource;
        int                         RegisterOffset;
        device_t                    Device;
} NAL_FREEBSD_PCI_ALLOCATION_SLOT;

typedef struct _NAL_FREEBSD_DMA_ALLOCATION_SLOT
{
        KVOID*                      KernelAddress;
        NAL_PHYSICAL_ADDRESS        PhysicalAddress;
        bus_dma_tag_t               DmaTag;
        bus_dmamap_t                DmaMap;
} NAL_FREEBSD_DMA_ALLOCATION_SLOT;

extern NAL_FREEBSD_PCI_ALLOCATION_SLOT    Global_PciAllocationSlotsTable[NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS];
extern NAL_FREEBSD_DMA_ALLOCATION_SLOT    Global_DmaAllocationSlotsTable[NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS];
#endif

NAL_STATUS
NalInitializeOs(VOID);

NAL_STATUS
NalExitOs(VOID);

KVOID*
NalKMemset(
    IN      KVOID*      Dest,
    IN      int         Value,
    IN      UINTN       Size
    );

VOID*
NalKtoUMemcpy(
    IN      VOID*       Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    );

KVOID*
NalKtoKMemcpy(
    IN      KVOID*      Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    );

KVOID*
NalUtoKMemcpy(
    IN      KVOID*      Dest,
    IN      VOID*       Source,
    IN      UINTN       Size
    );

BOOLEAN
_NalIsAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice
    );

#endif /* SDK_NAL_INC_FREEBSD_FREEBSDOS_I_H_ */
